// Service Worker (MV3) - Background logic with Local Bridge WS
// v2: Adds CDP proxy via chrome.debugger and forwards events/results back to Engine
// - Pairs with engine VM via WebSocket using a short-lived token
// - Accepts CDP commands over WS: cdp.attach / cdp.send / cdp.detach
// - For legacy flows: can dispatch prebuilt plans to content script

const STATE = {
  lastTabId: null,
  // ws: moved to content script; background no longer owns WS (fallback only)
  ws: null,
  token: null,
  bridgeOrigin: null,
  tokenSig: null,
};

// Fixed engine origin for the local bridge. In production, set this to your engine VM URL.
// We do NOT trust any origin coming from the web page.
const ENGINE_BRIDGE_ORIGIN = 'https://engine.snegikom.ru';
// Shared secret for signing the token. Must match ENGINE_BRIDGE_SECRET on the engine VM.
const ENGINE_BRIDGE_SECRET = 'fgdghfnrfufhu4hRT67_snegik89mdhJNFMnfh';

const DEV = !!(false || self.location?.hostname?.includes('localhost'));
function log(...args){ if (DEV) console.log('[ext]', ...args); }
function warn(...args){ console.warn('[ext]', ...args); }
function error(...args){ console.error('[ext]', ...args); }

// Connected content ports
const CONTENT_PORTS = new Set();

function sendToEngine(payload){
  for (const port of CONTENT_PORTS){
    try { port.postMessage({ type:'bridge_forward_to_engine', payload }); } catch {}
  }
}

// CDP session registry: sessionId -> { tabId }
const SESSIONS = new Map();
const SESSIONS_BY_TAB = new Map();

function genSessionId(){ return Math.random().toString(36).slice(2, 10); }

function wsUrlFromOrigin(origin) { return origin.replace(/^http/i, 'ws'); }

function buildBridgeUrl(origin, token, tokenSig) {
  const base = wsUrlFromOrigin(origin).replace(/\/$/, '');
  const params = new URLSearchParams({ token });
  if (tokenSig) params.set('sig', tokenSig);
  return `${base}/local-bridge/socket?${params.toString()}`;
}

// wsSend is deprecated in background; use sendToEngine via content ports
function wsSend(obj){
  if (CONTENT_PORTS.size > 0){
    sendToEngine(obj);
    return;
  }
  try { if (!STATE.ws) connectBridge(); } catch {}
  try { STATE.ws && STATE.ws.readyState===WebSocket.OPEN && STATE.ws.send(JSON.stringify(obj)); } catch {}
}


function connectBridge(){
  // Use fixed ENGINE_BRIDGE_ORIGIN; do not trust anything from the web page.
  const origin = ENGINE_BRIDGE_ORIGIN || STATE.bridgeOrigin;
  if (!origin || !STATE.token) return;
  // If content scripts are present, let them own the WS to engine
  if (CONTENT_PORTS.size > 0) return;
  const url = buildBridgeUrl(origin, STATE.token, STATE.tokenSig);
  try{
    STATE.ws = new WebSocket(url);
    STATE.ws.onopen = ()=>{
      console.log('[ext] Bridge WS connected');
      try { wsSend({ type:'hello', agent:'chrome', version:'1.1.0' }); } catch {}
    };
    STATE.ws.onmessage = async (ev)=>{
      try{
        const msg = JSON.parse(ev.data);
        await handleBridgeMessage(msg);
      }catch(e){ console.error('[ext] WS message error', e); }
    };
    STATE.ws.onclose = ()=>{
      console.log('[ext] Bridge WS closed');
      STATE.ws = null;
      // Auto-reconnect with backoff
      setTimeout(connectBridge, 2000);
    };
    STATE.ws.onerror = ()=>{
      try { STATE.ws && STATE.ws.close(); } catch {}
    };
  }catch(e){ console.error('[ext] Bridge connect error', e); }
}

async function handleBridgeMessage(msg){
  // CDP proxy interface only
  if (msg?.type === 'cdp.attach'){
    const id = msg.id;
    (async ()=>{
      try{
        const sessionInfo = await cdpAttach(msg);
        wsSend({ type:'cdp.result', id, ok:true, result: sessionInfo });
      }catch(e){ wsSend({ type:'cdp.result', id, ok:false, error: { message: String(e) } }); }
    })();
    return;
  }
  if (msg?.type === 'cdp.send'){
    const id = msg.id;
    (async ()=>{
      try{
        const result = await cdpSend(msg);
        wsSend({ type:'cdp.result', id, ok:true, result });
      }catch(e){ wsSend({ type:'cdp.result', id, ok:false, error: { message: String(e) } }); }
    })();
    return;
  }
  if (msg?.type === 'cdp.detach'){
    const id = msg.id;
    (async ()=>{
      try{
        await cdpDetach(msg);
        wsSend({ type:'cdp.result', id, ok:true, result: { detached: true } });
      }catch(e){ wsSend({ type:'cdp.result', id, ok:false, error: { message: String(e) } }); }
    })();
    return;
  }
}

async function ensureActiveTab(url) {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab || !tab.id) {
    const created = await chrome.tabs.create({ url: url || 'about:blank', active: true });
    return created.id;
  }
  if (url) {
    await chrome.tabs.update(tab.id, { url });
    await waitForTabComplete(tab.id);
  }
  return tab.id;
}

function waitForTabComplete(tabId) {
  return new Promise((resolve) => {
    function listener(updatedTabId, info) {
      if (updatedTabId === tabId && info.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        resolve(null);
      }
    }
    chrome.tabs.onUpdated.addListener(listener);
  });
}

function postProgress(event){ wsSend({ type:'event', ...event }); }

// Port messaging from content scripts
chrome.runtime.onConnect.addListener((port)=>{
  if (port?.name !== 'snegikom_bridge') return;
  CONTENT_PORTS.add(port);
  // If background WS is open, close it to avoid duplicate WS alongside content WS
  try { if (STATE.ws) { STATE.ws.close(); STATE.ws = null; } } catch {}
  port.onDisconnect.addListener(()=>{
    try { CONTENT_PORTS.delete(port); } catch {}
    // Если все content-скрипты отключились (например, пользователь ушёл с /dashboard
    // на произвольный сайт), поднимаем резервный WS из background, чтобы CDP‑мост
    // продолжал работать и не было 404: not_connected.
    if (CONTENT_PORTS.size === 0) {
      try { connectBridge(); } catch {}
    }
  });
  port.onMessage.addListener((m)=>{
    try{
      if (m && m.type==='bridge_forward' && m.payload){
        // Execute CDP commands and forward results to Engine via WS
        const p = m.payload;
        if (p.type==='cdp.attach'){
          (async ()=>{
            try { const res = await cdpAttach(p); wsSend({ type:'cdp.result', id:p.id, ok:true, result:res }); }
            catch(e){ wsSend({ type:'cdp.result', id:p.id, ok:false, error:{ message:String(e) } }); }
          })();
        } else if (p.type==='cdp.send'){
          (async ()=>{
            try { const res = await cdpSend(p); wsSend({ type:'cdp.result', id:p.id, ok:true, result:res }); }
            catch(e){ wsSend({ type:'cdp.result', id:p.id, ok:false, error:{ message:String(e) } }); }
          })();
        } else if (p.type==='cdp.detach'){
          (async ()=>{
            try { await cdpDetach(p); wsSend({ type:'cdp.result', id:p.id, ok:true, result:{ detached:true } }); }
            catch(e){ wsSend({ type:'cdp.result', id:p.id, ok:false, error:{ message:String(e) } }); }
          })();
        }
      }
    }catch{}
  });
});

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg?.type === 'pair'){
    // { token }
    STATE.token = msg.token;
    // Optional: compute token signature for backend validation when ENGINE_BRIDGE_SECRET is set
    if (ENGINE_BRIDGE_SECRET && typeof ENGINE_BRIDGE_SECRET === 'string') {
      // Simple HMAC-like placeholder: in real build you should replace this with a proper implementation
      // or precompute on the backend. Here we just keep a stub to show intent.
      try {
        // eslint-disable-next-line no-undef
        const data = `${msg.token}|${ENGINE_BRIDGE_SECRET}`;
        // Very simple hash substitute; replace at build-time or use chrome.crypto.subtle in future.
        STATE.tokenSig = btoa(unescape(encodeURIComponent(data))).slice(0, 32);
      } catch {
        STATE.tokenSig = null;
      }
    } else {
      STATE.tokenSig = null;
    }
    // Ensure WS connected as fallback when no content scripts are alive
    try { connectBridge(); } catch {}
    sendResponse?.({ ok:true });
    return true;
  }
});

// ========== CDP proxy via chrome.debugger ==========

async function cdpAttach(msg){
  // msg: { id, url?, tabId? }
  // Security: only allow attach to active tab or new tab created here
  let tabId = msg.tabId || null;
  if (!tabId){
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (activeTab && activeTab.id) {
      tabId = activeTab.id;
      // do not change the URL unless explicitly requested
      if (msg.url) {
        try { await chrome.tabs.update(tabId, { url: msg.url }); await waitForTabComplete(tabId); } catch {}
      }
    } else if (msg.url) {
      const created = await chrome.tabs.create({ url: msg.url, active: true });
      tabId = created.id;
      await waitForTabComplete(tabId);
    } else {
      // no active tab; create about:blank to attach
      const created = await chrome.tabs.create({ url: 'about:blank', active: true });
      tabId = created.id;
      await waitForTabComplete(tabId);
    }
  }

  try {
    await chrome.debugger.attach({ tabId }, '1.3');
    log('attach ok tab', tabId);
  } catch (e) {
    // Might already be attached; try to continue if so
    if (!String(e).includes('Another debugger')) throw e;
  }

  let sessionId = SESSIONS_BY_TAB.get(tabId);
  if (!sessionId){
    sessionId = genSessionId();
    SESSIONS.set(sessionId, { tabId });
    SESSIONS_BY_TAB.set(tabId, sessionId);
  }
  STATE.lastTabId = tabId;

  // Enable basic domains often required by agents
  try { await chrome.debugger.sendCommand({ tabId }, 'Page.enable', {}); } catch {}
  try { await chrome.debugger.sendCommand({ tabId }, 'DOM.enable', {}); } catch {}
  try { await chrome.debugger.sendCommand({ tabId }, 'Runtime.enable', {}); } catch {}
  try { await chrome.debugger.sendCommand({ tabId }, 'Network.enable', {}); } catch {}

  // Synthesize Target.attachedToTarget event for engine session manager compatibility
  try {
    const info = await chrome.tabs.get(tabId);
    const url = info?.url || 'about:blank';
    wsSend({
      type: 'cdp.event',
      session: sessionId,
      method: 'Target.attachedToTarget',
      params: {
        sessionId: sessionId,
        targetInfo: { targetId: `tab-${tabId}`, type: 'page', url },
        waitingForDebugger: false,
      }
    });
  } catch {}

  return { session: sessionId, tabId };
}

async function cdpSend(msg){
  const { session, method, params } = msg;
  const sessionRequired = !['Target.createTarget','Target.getTargets','Target.getTargetInfo','Target.setAutoAttach'].includes(method);
  if (sessionRequired && (!session || !SESSIONS.has(session))) throw new Error('Invalid session');
  const baseTabId = (session && SESSIONS.has(session)) ? SESSIONS.get(session).tabId : null;
  const { tabId } = { tabId: baseTabId };

  // Root-level shims when no session/tabId is available
  if (!session || !SESSIONS.has(session)){
    if (method === 'Target.getTargets'){
      try{
        const tabs = await chrome.tabs.query({});
        const infos = tabs.map(t => ({ targetId:`tab-${t.id}`, type:'page', url: t.url || 'about:blank', title: t.title || '' }));
        return { targetInfos: infos };
      }catch{ return { targetInfos: [] }; }
    }
    if (method === 'Target.getTargetInfo' && params && params.targetId){
      try{
        const tid = params.targetId;
        const idNum = String(tid).startsWith('tab-') ? Number(String(tid).slice(4)) : null;
        const tab = idNum ? await chrome.tabs.get(idNum) : null;
        return { targetInfo: { targetId: tid, type:'page', url: (tab && tab.url) || 'about:blank', title: (tab && tab.title) || '' } };
      }catch{ return { targetInfo: { targetId: params.targetId, type:'page', url:'about:blank', title:'' } }; }
    }
    if (method === 'Target.setAutoAttach'){
      return {};
    }
  }

  // Handle methods that need special translation
  if (method === 'Target.attachToTarget'){
    const tid = params?.targetId;
    const attachTabId = String(tid).startsWith('tab-') ? Number(String(tid).slice(4)) : tabId;
    try { await chrome.debugger.attach({ tabId: attachTabId }, '1.3'); } catch {}
    let sid = SESSIONS_BY_TAB.get(attachTabId);
    if (!sid){ sid = genSessionId(); SESSIONS_BY_TAB.set(attachTabId, sid); SESSIONS.set(sid, { tabId: attachTabId }); }
    // emit attached event
    try{
      const info = await chrome.tabs.get(attachTabId);
      wsSend({ type:'cdp.event', session: sid, method:'Target.attachedToTarget', params:{ sessionId: sid, targetInfo:{ targetId:`tab-${attachTabId}`, type:'page', url: info?.url || 'about:blank' }, waitingForDebugger:false }});
    }catch{}
    return { sessionId: sid };
  }
  if (method === 'Target.createTarget'){
    const url = params?.url || 'about:blank';
    // Open new tab next to the currently active tab and make it active
    let createOpts = { url, active: true };
    try {
      const [active] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (active && typeof active.index === 'number') createOpts.index = active.index + 1;
    } catch {}
    const created = await chrome.tabs.create(createOpts);
    const newTabId = created.id;
    await waitForTabComplete(newTabId);
    // attach and synthesize attach event
    try { await chrome.debugger.attach({ tabId: newTabId }, '1.3'); } catch {}
    let sid = SESSIONS_BY_TAB.get(newTabId);
    if (!sid){ sid = genSessionId(); SESSIONS_BY_TAB.set(newTabId, sid); SESSIONS.set(sid, { tabId: newTabId }); }
    try{
      const info = await chrome.tabs.get(newTabId);
      wsSend({ type:'cdp.event', session: sid, method:'Target.attachedToTarget', params:{ sessionId: sid, targetInfo:{ targetId:`tab-${newTabId}`, type:'page', url: info?.url || url }, waitingForDebugger:false }});
    }catch{}
    return { targetId: `tab-${newTabId}` };
  }
  if (method === 'Target.setAutoAttach'){
    // Not applicable in chrome.debugger context; return success
    return {};
  }
  if (method === 'Runtime.runIfWaitingForDebugger'){
    // Not applicable; return success
    return {};
  }
  if (method === 'Target.activateTarget'){
    // Activate a target by focusing the tab
    try{
      const tid = params?.targetId;
      const tId = String(tid).startsWith('tab-') ? Number(String(tid).slice(4)) : tabId;
      await chrome.tabs.update(tId, { active: true });
      return {};
    }catch(e){ return {}; }
  }

  const result = await chrome.debugger.sendCommand({ tabId }, method, params || {});
  // Provide minimal shims for Target.* methods used by engine when chrome.debugger doesn't return full info
  if (method === 'Target.getTargets'){
    try {
      const tab = await chrome.tabs.get(tabId);
      const targetId = `tab-${tabId}`;
      return { targetInfos: [{ targetId, type:'page', url: tab.url || 'about:blank', title: tab.title || '' }] };
    } catch {
      return { targetInfos: [] };
    }
  }
  if (method === 'Target.getTargetInfo' && params && params.targetId){
    try{
      const tid = params.targetId;
      const idStr = String(tid).startsWith('tab-') ? Number(String(tid).slice(4)) : tabId;
      const tab = await chrome.tabs.get(idStr);
      return { targetInfo: { targetId: tid, type:'page', url: tab.url || 'about:blank', title: tab.title || '' } };
    }catch{
      return { targetInfo: { targetId: params.targetId, type:'page', url:'about:blank', title:'' } };
    }
  }
  return result || {};
}

async function cdpDetach(msg){
  const { session } = msg;
  if (!session || !SESSIONS.has(session)) return;
  const { tabId } = SESSIONS.get(session);
  try { await chrome.debugger.detach({ tabId }); } catch {}
  SESSIONS.delete(session);
  SESSIONS_BY_TAB.delete(tabId);
}

chrome.debugger.onEvent.addListener((source, method, params)=>{
  try{
    const tabId = source?.tabId;
    if (!tabId) return;
    const session = SESSIONS_BY_TAB.get(tabId);
    if (!session) return;
    // Map events to include target info where expected
    if (method === 'Target.attachedToTarget' || method === 'Target.detachedFromTarget'){
      // ensure params structure contains sessionId and targetInfo
      const safe = Object.assign({ sessionId: session, targetInfo: { targetId:`tab-${tabId}`, type:'page' }}, params||{});
      wsSend({ type:'cdp.event', session, method, params: safe });
      return;
    }
    // forward everything else
    wsSend({ type:'cdp.event', session, method, params });
  }catch(e){ /* ignore */ }
});

chrome.debugger.onDetach.addListener((source, reason)=>{
  try{
    const tabId = source?.tabId;
    const session = tabId ? SESSIONS_BY_TAB.get(tabId) : null;
    if (session){
      SESSIONS.delete(session);
      SESSIONS_BY_TAB.delete(tabId);
      // Emit Target.detachedFromTarget to match SessionManager expectation
      wsSend({ type:'cdp.event', session, method:'Target.detachedFromTarget', params:{ sessionId: session, targetId: `tab-${tabId}`, reason } });
    }
  }catch(e){ /* ignore */ }
});
